<?php
// FILE: create_delivery_note.php
require_once __DIR__ . '/init.php';

// 🛑 التحقق من صلاحية مندوب التسليم فقط
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'delivery_agent') {
    header('Location: ' . $config['base_url'] . 'login.php');
    exit;
}

$username = htmlspecialchars($_SESSION['user']['username']);
$user_id = $_SESSION['user']['id']; 
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>تسجيل سند تسليم جديد</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
    body{font-family:'Tajawal',sans-serif;background:#e6f3ff;padding:16px;direction:rtl;}
    .card{border-radius:12px;padding:16px;margin-bottom:16px;box-shadow:0 4px 12px rgba(0,0,0,0.08);}
    .product-item { border: 1px solid #ccc; border-radius: 8px; padding: 15px; margin-bottom: 10px; background-color: #f8f9fa; }
    .disabled-input { background-color: #e9ecef !important; }
    
    /* إلغاء القيود السابقة من #video للسيطرة عبر الحاوية */
    #video{
        border-radius:8px;
        width:100%;
        height:100%;
        margin-bottom:25px;
        display:none;
    }

    /* التحكم في حجم الحاوية (حل مشكلة ارتفاع الفيديو السابق) */
    #video-wrapper {
        max-width: 80%;
        max-height: 250px; /* الارتفاع الذي تريده */
        overflow: hidden;
        margin: 0 auto;
        position: relative;
    }

    /* التعديل التصميمي لتحسين تركيز الكاميرا */
    .barcode-video-frame {
        border: 3px solid #007bff; 
        border-radius: 10px;
        box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
    }

    /* 🚀 الحل المباشر لزر الإدخال (السهم المعقوف) */
    /* يُطبق على حقل رقم السند وحقول الباركود */
    input[type="text"], input[type="search"] {
        direction: ltr; /* فرض الاتجاه من اليسار لليمين لتفعيل زر Enter/Go */
        text-align: right; /* الحفاظ على محاذاة النص المدخل إلى اليمين */
    }
</style>
</head>
<body>
    <div class="container mt-5">
        <div class="card bg-white p-4">
            <h1 class="text-center text-primary mb-4">تسجيل سند تسليم جديد</h1>
            <hr>
            
            <div id="statusMessage" class="mt-3 mb-3"></div>

            <form id="multiDeliveryForm">
                
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label for="delivery_note_ref" class="form-label">رقم سند التسليم (مطلوب)</label>
                        <input type="text" class="form-control" id="delivery_note_ref" name="delivery_note_ref" required>
                    </div>
                    <div class="col-md-6 mb-3">
                        <label for="delivery_date" class="form-label">تاريخ التسليم الفعلي (مطلوب)</label>
                        <input type="date" class="form-control" id="delivery_date" name="delivery_date" required>
                    </div>
                </div>
                
                <hr>

                <h5 class="mb-3 text-info">المنتجات المراد تسليمها</h5>
                <div id="productsContainer">
                    </div>
                
                <button type="button" class="btn btn-sm btn-outline-secondary mb-4" id="addProductBtn">
                    ➕ إضافة منتج آخر
                </button>
                <div id="scanArea" class="mb-3">
                    <button class="btn btn-info text-white mb-3" id="scanBtn">مسح بالكميرا</button>
                    <div id="video-wrapper">
                        <video id="video" autoplay></video>
                    </div>
                </div>

                <div class="d-flex justify-content-between mt-4">
                    <button type="submit" class="btn btn-success btn-lg" id="saveBtn">حفظ السند</button>
                    <button type="button" class="btn btn-warning btn-lg" id="newNoteBtn" style="display:none;">سند جديد</button> 
                    <a href="/barcode-system/delivery_dashboard.php" class="btn btn-secondary btn-lg">العودة للوحة المندوب</a>
                </div>
            </form>
        </div>
    </div>

<script src="https://unpkg.com/@zxing/library@0.18.6/umd/index.min.js"></script>
<script>
    const productsContainer = document.getElementById('productsContainer');
    const addProductBtn = document.getElementById('addProductBtn');
    const multiDeliveryForm = document.getElementById('multiDeliveryForm');
    const saveBtn = document.getElementById('saveBtn');
    const newNoteBtn = document.getElementById('newNoteBtn');
    const statusDiv = document.getElementById('statusMessage');
    const video = document.getElementById('video');
    const videoWrapper = document.getElementById('video-wrapper'); // الحصول على الحاوية الجديدة
    const scanBtn = document.getElementById('scanBtn');
    const API_URL = '/barcode-system/api/delivery.php';
    const SEARCH_URL = '/barcode-system/api/products.php';
    let productCounter = 0;
    let productsData = {}; 
    let scanActive = false;
    
    // 💡 تقليل زمن التأخير بين محاولات الفحص
    const codeReader = new ZXing.BrowserBarcodeReader(null, {
        delayBetweenScanAttempts: 200 
    });


    // ------------------ الأدوات المساعدة والتنقية ------------------
    function showStatus(message, type = 'info') {
        statusDiv.innerHTML = `<div class="alert alert-${type}">${message}</div>`;
    }
    
    // دالة تنقية الباركود
    function cleanBarcode(barcode){
        if (typeof barcode !== 'string') return '';
        return barcode.replace(/[\x00-\x1F\x7F]+/g, '').trim();
    }

    // الدالة المخصصة للتحقق من تكرار الباركود في النموذج الحالي
    function isBarcodeDuplicate(barcode, currentInputId = null) {
        const cleanedBarcode = cleanBarcode(barcode);
        if (!cleanedBarcode) return false;

        let duplicateCount = 0;
        
        for (const input of document.querySelectorAll('.barcode-input')) {
            if (currentInputId !== null && input.id === currentInputId) {
                continue;
            }

            const storedCleanedBarcode = cleanBarcode(input.value);

            if (storedCleanedBarcode === cleanedBarcode) {
                duplicateCount++;
            }
        }
        
        return duplicateCount > 0;
    }

    // ------------------ 1. إضافة حقول المنتج ------------------
    function addProductItem(barcodeValue = '') {
        const trimmedBarcode = barcodeValue.trim();
        
        if (trimmedBarcode && isBarcodeDuplicate(trimmedBarcode, null)) {
            showStatus(`تنبيه: الباركود ${trimmedBarcode} مضاف بالفعل إلى هذا السند!`, 'warning');
            return;
        }

        const index = productCounter++;
        const itemHtml = `
            <div class="product-item" data-index="${index}">
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label for="barcode-${index}" class="form-label">الباركود</label>
                        <div class="input-group">
                            <input type="search" class="form-control barcode-input" id="barcode-${index}" placeholder="أدخل الباركود أو امسح" value="${trimmedBarcode}" required>
                            <button type="button" class="btn btn-danger remove-product-btn" data-index="${index}" title="إزالة المنتج" ${productCounter === 1 ? 'style="display:none;"' : ''}>🗑️</button>
                        </div>
                    </div>
                    <div class="col-md-6 mb-3">
                        <label for="desc-${index}" class="form-label">مواصفات المنتج (الاسم والمقاس والتواريخ)</label>
                        <textarea class="form-control product-desc-input disabled-input" id="desc-${index}" rows="3" readonly></textarea>
                        <input type="hidden" class="product-id" name="products[${index}][id]">
                    </div>
                </div>
            </div>
        `;
        productsContainer.insertAdjacentHTML('beforeend', itemHtml);
        
        document.getElementById(`barcode-${index}`).addEventListener('change', (e) => {
            const newBarcode = e.target.value.trim();
            const inputId = e.target.id;

            if (newBarcode && isBarcodeDuplicate(newBarcode, inputId)) {
                showStatus(`تنبيه: الباركود ${newBarcode} مضاف بالفعل إلى حقل آخر في هذا السند!`, 'warning');
                e.target.value = '';
                document.getElementById(`desc-${index}`).value = '';
                delete productsData[index]; 
                return;
            }
            fetchProduct(newBarcode, index);
        });
        
        document.querySelectorAll('.remove-product-btn').forEach(btn => {
            btn.style.display = (productCounter > 1) ? 'block' : 'none';
        });

        if (trimmedBarcode) {
            fetchProduct(trimmedBarcode, index);
        }
    }

    // ------------------ 2. وظيفة البحث بالباركود (لتحقق تكرار ID) ------------------
    async function fetchProduct(barcode, index) {
        const cleanedBarcode = cleanBarcode(barcode);
        const descField = document.getElementById(`desc-${index}`);
        const idField = document.querySelector(`.product-item[data-index="${index}"] .product-id`);
        descField.value = 'جاري البحث...';
        idField.value = '';

        delete productsData[index];

        try {
            const res = await fetch(SEARCH_URL, {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({ barcode: cleanedBarcode }) 
            });
            const j = await res.json();

            if (j.error || !j.id) {
                descField.value = `❌ ${j.error || 'المنتج غير موجود'}`;
                showStatus(`لم يتم العثور على المنتج بالباركود ${barcode}`, 'danger');
                return;
            }

            const newProductId = j.id;
            let isIdDuplicate = false;
            for (const key in productsData) {
                if (productsData[key].id === newProductId) {
                    isIdDuplicate = true;
                    break;
                }
            }

            if (isIdDuplicate) {
                descField.value = `❌ مكرر (المنتج ID: ${newProductId}). مضاف بالفعل في هذا السند.`;
                showStatus(`تنبيه: المنتج ${j.name} (ID: ${newProductId}) مضاف بالفعل إلى هذا السند!`, 'warning');
                return; 
            }

            if (j.delivery_note_ref && j.delivery_note_ref.trim() !== '') {
                descField.value = `❌ مسجل مسبقاً (السند: ${j.delivery_note_ref}). التعديل مقتصر على المدير.`;
                showStatus(`المنتج ${j.name} مُسجل له سند بالفعل.`, 'warning');
                return;
            }

            const description = `الاسم: ${j.name}\nالمقاس: ${j.size || 'غير محدد'}\nتاريخ الإنتاج: ${j.production_date || 'غير محدد'}\nتاريخ الانتهاء: ${j.expiry_date || 'غير محدد'}`;
            descField.value = description;
            idField.value = newProductId; 

            productsData[index] = { id: newProductId }; 

            showStatus(`✅ تم تحميل بيانات المنتج ${j.name} (ID: ${newProductId})`, 'success');

        } catch (e) {
            descField.value = 'خطأ في الاتصال.';
            showStatus('حدث خطأ في الاتصال: ' + e.message, 'danger');
        }
    }

    // ------------------ 3. وظيفة الحفظ (إرسال السند المتعدد) ------------------
    multiDeliveryForm.addEventListener('submit', async e => {
        e.preventDefault();

        const ref = document.getElementById('delivery_note_ref').value.trim();
        const date = document.getElementById('delivery_date').value;

        if (!ref || !date) {
            showStatus('يرجى إدخال رقم السند وتاريخ التسليم.', 'danger');
            return;
        }

        const productsToSave = [];
        for (const index in productsData) {
            if (productsData[index].id) {
                productsToSave.push(productsData[index].id);
            }
        }
        
        const uniqueProductsToSave = [...new Set(productsToSave)];
        
        const productItems = document.querySelectorAll('.product-item');
        let incompleteCount = 0;
        
        productItems.forEach(item => {
            const barcodeInput = item.querySelector('.barcode-input');
            const productIndex = item.getAttribute('data-index');

            if (barcodeInput && barcodeInput.value.trim() !== '' && !productsData[productIndex]) {
                 incompleteCount++;
            }
        });


        if (uniqueProductsToSave.length === 0) {
            showStatus('لم يتم التحقق من أي باركود بنجاح. يرجى مراجعة الحقول.', 'danger');
            return;
        }

        if (incompleteCount > 0) {
             showStatus(`تحذير: يوجد ${incompleteCount} منتجات لم يتم التحقق من باركودها بنجاح. سيتم حفظ المنتجات الصحيحة فقط.`, 'warning');
        }
        
        showStatus(`جاري حفظ السند رقم ${ref} لعدد ${uniqueProductsToSave.length} منتجات...`, 'info');
        
        saveBtn.disabled = true;
        document.getElementById('delivery_note_ref').readOnly = true;
        document.getElementById('delivery_date').readOnly = true;
        addProductBtn.disabled = true;

        let successCount = 0;
        let errorCount = 0;
        let lastError = '';

        for (const productId of uniqueProductsToSave) {
            const data = {
                id: productId,
                delivery_note_ref: ref,
                delivery_date: date
            };
            
            try {
                const res = await fetch(API_URL, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(data) 
                });
                const j = await res.json();

                if (j.ok) {
                    successCount++;
                } else {
                    errorCount++;
                    lastError = j.error;
                }
            } catch (e) {
                errorCount++;
                lastError = e.message;
            }
        }
        
        if (errorCount === 0) {
            showStatus(`✅ تم حفظ السند رقم ${ref} لجميع المنتجات (${successCount} منتج) بنجاح.`, 'success');
            document.querySelectorAll('.barcode-input').forEach(input => input.readOnly = true);
            document.querySelectorAll('.remove-product-btn').forEach(btn => btn.style.display = 'none');
            saveBtn.style.display = 'none'; 
            newNoteBtn.style.display = 'block'; 
            
        } else {
            let finalErrorMsg = `❌ فشل حفظ ${errorCount} منتجات. تم حفظ ${successCount} منتج بنجاح.`;
            if (lastError) {
                 finalErrorMsg += ` آخر خطأ: ${lastError}`;
            }
            showStatus(finalErrorMsg, 'danger');
            
            saveBtn.disabled = false;
            document.getElementById('delivery_note_ref').readOnly = false;
            document.getElementById('delivery_date').readOnly = false;
            addProductBtn.disabled = false;
        }
    });

    // ------------------ 4. أزرار الإجراءات ------------------
    newNoteBtn.addEventListener('click', () => {
        window.location.reload(); 
    });

    addProductBtn.addEventListener('click', () => addProductItem());
    
    productsContainer.addEventListener('click', (e) => {
        if (e.target.classList.contains('remove-product-btn')) {
            const item = e.target.closest('.product-item');
            if (item) {
                const indexToRemove = item.getAttribute('data-index');
                item.remove();
                delete productsData[indexToRemove]; 
                
                if (document.querySelectorAll('.product-item').length === 1) {
                        document.querySelector('.remove-product-btn').style.display = 'none';
                }
            }
        }
    });

    // ------------------ 5. وظيفة المسح الضوئي (مُحسَّنة) ------------------
    
    // const scanBtnIcon = document.getElementById('scanBtn').querySelector('i'); // هذا السطر قد يسبب خطأ إذا لم يكن هناك وسم <i>

    scanBtn.addEventListener('click', () => {
        if (scanActive) {
            codeReader.reset();
            video.style.display = 'none';
            videoWrapper.classList.remove('barcode-video-frame'); // إزالة فئة CSS
            scanBtn.textContent = 'مسح بالكميرا';
            scanActive = false;
            return;
        }

        video.style.display = 'block';
        videoWrapper.classList.add('barcode-video-frame'); // إضافة فئة CSS على الحاوية
        scanBtn.textContent = 'إيقاف الكاميرا';
        scanActive = true;

        // فرض الدقة والكاميرا الخلفية
        codeReader.decodeFromConstraints({
            video: {
                width: { ideal: 640 }, 
                height: { ideal: 480 },
                facingMode: "environment" 
            }
        }, video, (result, err) => {
            if (result) {
                const scannedBarcode = result.text.trim();
                
                if (isBarcodeDuplicate(scannedBarcode, null)) {
                    showStatus(`تنبيه: الباركود ${scannedBarcode} مضاف بالفعل إلى هذا السند!`, 'warning');
                } else {
                    addProductItem(scannedBarcode);
                }
                
                codeReader.reset();
                video.style.display = 'none';
                videoWrapper.classList.remove('barcode-video-frame');
                scanBtn.textContent = 'مسح بالكميرا';
                scanActive = false;
            }

            if (err && !(err instanceof ZXing.NotFoundException)) {
                console.error(err);
            }
        }).catch(err => {
            console.error("خطأ في تشغيل الكاميرا: ", err);
            codeReader.reset();
            video.style.display = 'none';
            videoWrapper.classList.remove('barcode-video-frame');
            scanBtn.textContent = 'مسح بالكميرا';
            scanActive = false;
        });
    });

    // تهيئة الواجهة
    addProductItem();
</script>
</body>
</html>