<?php
// FILE: manage_delivery_notes.php - خاص بالسوبر أدمن (مُعدَّل لإضافة زر الحذف)
require_once __DIR__ . '/../init.php';

// 🛑 التعديل هنا: السماح لـ super_admin و admin بالدخول
if (!isset($_SESSION['user']) || ($_SESSION['user']['role'] !== 'super_admin' && $_SESSION['user']['role'] !== 'admin')) {
    header('Location: ' . $config['base_url'] . 'login.php');
    exit;
}

// 🎯 تحديد ما إذا كان المستخدم سوبر أدمن أم لا (لتطبيق قيد التعديل والحذف)
$is_super_admin = ($_SESSION['user']['role'] === 'super_admin');

// 1. جلب أرقام السندات الفريدة لجميع المندوبين
// تم إزالة المسافات البادئة من داخل الاستعلام لتجنب مشاكل النسخ
$stmt = $pdo->prepare("
SELECT
p.delivery_note_ref,
MIN(p.delivery_date) AS delivery_date,
COUNT(p.id) AS product_count,
MAX(u.username) AS agent_username
FROM
products p
JOIN
users u ON p.delivery_recorded_by = u.id
WHERE
p.delivery_note_ref IS NOT NULL AND p.delivery_note_ref != ''
GROUP BY
p.delivery_note_ref
ORDER BY
delivery_date DESC, p.delivery_note_ref DESC
");
$stmt->execute();
$delivery_notes = $stmt->fetchAll(PDO::FETCH_ASSOC);

$username = htmlspecialchars($_SESSION['user']['username']);
?>

<!doctype html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>إدارة سندات التسليم</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
    <style>
        body { font-family: 'Tajawal', sans-serif; background: #e9ecef; direction: rtl; }
        .card { border-radius: 15px; box-shadow: 0 4px 15px rgba(0,0,0,0.1); }
    </style>
</head>
<body>
    <div class="container mt-5">
        <div class="card p-4">
            <h1 class="text-center text-danger mb-4"> إدارة السندات</h1>
            <p class="lead text-center">قائمة بجميع سندات التسليم في النظام.</p>
            <hr>

            <div id="statusMessage" class="mt-3"></div>

            <div class="d-flex justify-content-between mb-3">
                 <a href="/barcode-system/admin_dashboard.php" class="btn btn-secondary">العودة للوحة التحكم</a>
                <span class="badge bg-primary fs-6">عدد السندات: <?= count($delivery_notes) ?></span>
            </div>
            
            <?php if (empty($delivery_notes)): ?>
                <div class="alert alert-warning text-center">
                    لا توجد سندات تسليم مسجلة في النظام بعد.
                </div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>رقم السند</th>
                                <th>مندوب التسليم</th> 
                                <th>عدد المنتجات</th>
                                <th>تاريخ التسليم</th>
                                <th>الإجراء</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($delivery_notes as $note): ?>
                            <tr>
                                <td class="fw-bold text-success"><?= htmlspecialchars($note['delivery_note_ref']) ?></td>
                                <td><?= htmlspecialchars($note['agent_username']) ?></td> 
                                <td><?= $note['product_count'] ?></td>
                                <td><?= htmlspecialchars($note['delivery_date']) ?></td>
                                <td>
                                    <button 
                                        class="btn btn-sm btn-info text-white" 
                                        onclick="viewDetails('<?= htmlspecialchars($note['delivery_note_ref']) ?>')"
                                    >
                                        عرض التفاصيل
                                    </button>
                                    
                                    <?php if ($is_super_admin): ?>
                                    <button class="btn btn-danger btn-sm delete-note-btn ms-2" 
                                        data-ref="<?= htmlspecialchars($note['delivery_note_ref']) ?>" 
                                        title="حذف السند نهائياً">
                                        <i class="bi bi-trash"></i> حذف
                                    </button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
        // ⚠️ تأكد من صحة مسار ملف API الإدارة بالنسبة لملف manage_delivery_notes.php
        const DELETE_API_URL = '/barcode-system/admin/delivery_management.php';
        document.addEventListener('DOMContentLoaded', () => {
            // تفويض الحدث لزر الحذف
            document.body.addEventListener('click', function(e) {
                const deleteBtn = e.target.closest('.delete-note-btn');
                if (deleteBtn) {
                    const ref = deleteBtn.getAttribute('data-ref');
                    handleDeleteDeliveryNote(ref);
                }
            });
        });

        function handleDeleteDeliveryNote(ref) {
            // إذا لم يكن المستخدم سوبر أدمن، لن تعمل هذه الدالة (لكن الأمان الأساسي في PHP/API)
            if (!confirm(`تحذير: هل أنت متأكد من حذف سند التسليم رقم ${ref}؟ سيتم إزالة رقم السند من جميع المنتجات المرتبطة به.`)) {
                return;
            }

            const statusElement = document.getElementById('statusMessage');
            if (statusElement) {
                statusElement.innerHTML = `<div class="alert alert-info">جاري حذف السند ${ref}...</div>`;
            }

            fetch(DELETE_API_URL, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ delivery_note_ref: ref })
            })
            .then(response => {
                if (!response.ok) {
                    return response.json().then(error => { throw new Error(error.error || `HTTP error! status: ${response.status}`); });
                }
                return response.json();
            })
            .then(data => {
                // نجاح الحذف
                if (statusElement) {
                    statusElement.innerHTML = `<div class="alert alert-success">✅ تم حذف السند ${ref} بنجاح. تم تحديث ${data.updated_products} منتج.</div>`;
                }
                // إعادة تحميل الصفحة
                setTimeout(() => {
                    window.location.reload(); 
                }, 1500); 
            })
            .catch(error => {
                // فشل الحذف
                console.error('Delete error:', error);
                if (statusElement) {
                    statusElement.innerHTML = `<div class="alert alert-danger">❌ فشل الحذف: ${error.message}</div>`;
                }
            });
        }
        
        // الدالة الأصلية لعرض التفاصيل
        function viewDetails(noteRef) {
            // بما أن manage_delivery_notes.php في مجلد admin (افتراضيا)، نستخدم '../' للعودة لمجلد الجذر
            const detailPageUrl = '../view_note_details.php?note_ref=' + encodeURIComponent(noteRef);
            window.location.href = detailPageUrl;
        }
    </script>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>