<?php
require_once __DIR__ . '/../init.php';

// 1. التحقق من صلاحيات الدخول: مسموح لـ super_admin و admin
if (!isset($_SESSION['user']) || ($_SESSION['user']['role'] !== 'super_admin' && $_SESSION['user']['role'] !== 'admin')) {
    header('Location: /barcode-system/admin/login.php');
    exit;
}

// 🛑 تحديد الدور وتطبيق القيد والقيمة الافتراضية
$user_role = $_SESSION['user']['role'] ?? 'admin'; 

// إذا كان المستخدم هو 'admin' (وليس super_admin)، اجعل الحقل للقراءة فقط
$production_date_readonly = ($user_role === 'admin') ? 'readonly' : ''; 

// القيمة الافتراضية لتاريخ الإنتاج هي تاريخ اليوم
$default_production_date = date('Y-m-d'); 


// 🛑 إعدادات الترقيم (Pagination) 🛑
$products_per_page = 10;
$current_page = max(1, intval($_GET['page'] ?? 1)); // تأكد من أن الصفحة لا تقل عن 1

// 🎯 (أ) جلب العدد الكلي للمنتجات
$total_products_stmt = $pdo->query("SELECT COUNT(*) FROM products");
$total_products = $total_products_stmt->fetchColumn();

// 🎯 (ب) حساب عدد الصفحات الكلي
$total_pages = ceil($total_products / $products_per_page);

// 🎯 (ت) حساب نقطة البداية (OFFSET)
$start_from = ($current_page - 1) * $products_per_page;
if($start_from < 0) $start_from = 0; // تأمين ضد القيم السالبة

// 🎯 (ث) استعلام جلب المنتجات المحدود (مع الترتيب والتحديد)
$rows = $pdo->query("
    SELECT *, delivery_note_ref, delivery_date 
    FROM products 
    ORDER BY created_at DESC 
    LIMIT $start_from, $products_per_page
")->fetchAll(PDO::FETCH_ASSOC);

// التعامل مع إضافة المنتج عبر POST
$error = $success = '';
if($_SERVER['REQUEST_METHOD']==='POST'){
    $barcode = trim($_POST['barcode'] ?? '');
    $name = trim($_POST['name'] ?? '');
    $size = trim($_POST['size'] ?? '');
    $production_date = $_POST['production_date'] ?: null;
    $notes = trim($_POST['notes'] ?? '');

    // التحقق
    if($_SESSION['user']['role'] !== 'super_admin' && $_SESSION['user']['role'] !== 'admin'){
        $error = "غير مصرح لك بإضافة منتجات.";
    } elseif(!$barcode || !$name){
        $error="الباركود واسم المنتج مطلوبان.";
    } else {
        // التأكد من عدم وجود الباركود مسبقاً
        $check = $pdo->prepare("SELECT id FROM products WHERE barcode=?");
        $check->execute([$barcode]);
        if($check->fetch()){
            $error="رمز الباركود مسجل مسبقاً.";
        } else {
            try {
                $stmt = $pdo->prepare("INSERT INTO products (barcode, name, size, production_date, notes, created_by) VALUES (?,?,?,?,?,?)");
                $stmt->execute([$barcode, $name, $size ?: null, $production_date, $notes, $_SESSION['user']['id']]);
                $success = "تمت إضافة المنتج بنجاح. معرف: " . $pdo->lastInsertId();
                // إعادة توجيه لمنع إرسال النموذج مرة أخرى عند تحديث الصفحة
                header("Location: " . $_SERVER['PHP_SELF'] . "?success=" . urlencode($success));
                exit;
            } catch (Exception $e) {
                $error = "حدث خطأ أثناء الحفظ: " . $e->getMessage();
            }
        }
    }
}

// عرض رسائل النجاح أو الخطأ بعد إعادة التوجيه
if(isset($_GET['success'])){
    $success = htmlspecialchars($_GET['success']);
}
?>
<!doctype html>
<html lang="ar">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>لوحة الإدارة - Backcare</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
<style>
    body{font-family:Tajawal,sans-serif;background:#7fb8ffa;padding:16px;direction:rtl;}
    .card{border-radius:12px;padding:16px;margin-bottom:16px;box-shadow:0 4px 12px rgba(0,0,0,0.05);}
    .table-responsive{overflow-x:auto;}
    
    /* 🟢 تنسيقات الألوان الجديدة لحقول الإدخال */
    .form-control {
        /* لون خلفية فاتح جداً لتمييز الحقول */
        background-color: #d9eaff !important; 
        /* 💡 تم إضافة هذا السطر لإظهار الحدود بشكل دائم بلون رمادي فاتح */
        border: 1px solid #d1d5db; 
        transition: border-color 0.2s;
    }
    .form-control:focus {
        /* تركيز خفيف عند الضغط على الحقل */
        border-color: #a0c3ff; 
        box-shadow: 0 0 0 0.25rem rgba(0, 123, 255, 0.1);
    }
    
    /* 🟢 تنسيق الـ input-group لـ RTL لظهور الأزرار على اليسار */
    /* التنسيق الأساسي للحقول المصاحبة بزر واحد */
    .input-group .form-control { 
        /* الزاوية اليمنى (البداية) مدورة */
        border-top-right-radius: 0.375rem !important; 
        border-bottom-right-radius: 0.375rem !important;
        /* الزاوية اليسرى (النهاية) غير مدورة */
        border-top-left-radius: 0 !important;
        border-bottom-left-radius: 0 !important;
        text-align: right; 
    }
    /* الزر المصاحب (في أقصى اليسار) يجب أن تكون زاويته مدورة */
    .input-group .btn-submit-box { 
        border-top-left-radius: 0.375rem !important; 
        border-bottom-left-radius: 0.375rem !important;
    }

    /* 🟢 تنسيق حقل الباركود (يحتوي على زرين) */
    .input-group #barcodeInput {
        /* الحقل يحتوي على زرين، لذا زواياه اليسرى يجب أن لا تكون مدورة */
        border-top-left-radius: 0 !important;
        border-bottom-left-radius: 0 !important;
    }
    /* زر السهم المعقوف (الآن في المنتصف) يجب أن يكون بدون زوايا */
    .input-group .btn-barcode-submit {
         border-radius: 0 !important; 
         /* لون أزرق هادئ */
         background-color: #4a7dce !important; 
         border-color: #4a7dce !important;
    }
    /* زر الكاميرا (الآن في أقصى اليسار) يجب أن تكون زاويته مدورة */
    .input-group #scanBtn { 
        border-top-left-radius: 0.375rem !important; 
        border-bottom-left-radius: 0.375rem !important;
        /* لون أخضر هادئ */
        background-color: #209353 !important;
        border-color: #209353 !important;
    }
    /* تعديل الزرار المصاحب لبقية الحقول ليتناسب مع درجة لون السهم */
    .input-group .btn-submit-box {
        background-color: #4a7dce !important; 
        border-color: #4a7dce !important;
    }
    
</style>
</head>
<body>
    <div class="container">
        
        <div class="mb-3">
    <h2 class="h4 mb-3">لوحة الإدارة</h2>
    <div class="d-grid gap-2 d-md-flex justify-content-md-start mb-3">
        
        <?php if ($_SESSION['user']['role'] === 'super_admin'): ?> 
            <a href="/barcode-system/admin/manage_delivery_notes.php" class="btn btn-primary">📋 إدارة السندات</a>
            
            <a href="/barcode-system/admin/generate_barcode_images.php" class="btn btn-info text-white">🖨️ طباعة المسجل</a>
            
            <a href="/barcode-system/admin/generate_temporary_barcodes.php" class="btn btn-success">📄 طباعة (100K+)</a>
        
            <a href="/barcode-system/admin/manage_users.php" class="btn btn-success">➕ إدارة المستخدمين</a>
        
            <a href="/barcode-system/admin/export_csv.php" class="btn btn-secondary">📥 تصدير CSV</a>
        <?php endif; ?>

        <a href="/barcode-system/admin/change_password.php" class="btn btn-warning">تغيير كلمة المرور</a>
        <a href="/barcode-system/auth.php?action=logout" class="btn btn-danger">تسجيل خروج</a>
    </div>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger"><?= $error ?></div>
<?php endif; ?>
<?php if ($success): ?>
    <div class="alert alert-success"><?= $success ?></div>
<?php endif; ?>

<div class="card">
    <div class="d-flex justify-content-end mb-3">
        <a href="/barcode-system/admin_dashboard.php" class="btn btn-primary btn-sm">
            <i class="bi bi-house-door-fill me-2"></i> العودة للوحة التحكم
        </a>
    </div>
    
    <h4>إضافة منتج جديد</h4>
    <?php if ($_SESSION['user']['role'] === 'super_admin' || $_SESSION['user']['role'] === 'admin'): ?>
    <form method="post" id="addProductForm">
        <div class="row">
            
            <div class="col-12 col-md-6 mb-2">
                <label class="form-label">باركود المنتج (مطلوب)</label>
                <div class="input-group">
                    
                    <input name="barcode" id="barcodeInput" class="form-control" placeholder="أدخل الباركود يدوياً" required value="<?= htmlspecialchars($_POST['barcode'] ?? '') ?>">
                    
                    <button type="submit" class="btn btn-primary btn-barcode-submit" title="إدخال الباركود يدوياً">
                        <i class="bi bi-arrow-return-left"></i>
                    </button>
                    
                    <button type="button" id="scanBtn" class="btn btn-success" title="مسح الباركود بالكاميرا">
                        <i class="bi bi-camera-fill"></i>
                    </button>
                    
                </div>
                
                <video id="video" width="100%" height="auto" autoplay style="border-radius:8px; display:none; max-width: 320px;"></video>
            </div>
            
            <div class="col-md-6 mb-2">
                <label class="form-label">اسم المنتج (مطلوب)</label>
                <div class="input-group">
                    <input name="name" class="form-control" required value="<?= htmlspecialchars($_POST['name'] ?? '') ?>">
                    <button type="submit" class="btn btn-primary btn-submit-box" title="حفظ البيانات">
                        <i class="bi bi-arrow-return-left"></i>
                    </button>
                </div>
            </div>
            
            <div class="col-md-6 mb-2">
                <label class="form-label">المقاس (Size)</label>
                <div class="input-group">
                    <input name="size" class="form-control" value="<?= htmlspecialchars($_POST['size'] ?? '') ?>">
                    <button type="submit" class="btn btn-primary btn-submit-box" title="حفظ البيانات">
                        <i class="bi bi-arrow-return-left"></i>
                    </button>
                </div>
            </div>
            
            <div class="col-md-6 mb-2">
                <label class="form-label">تاريخ الإنتاج</label>
                <div class="input-group">
                    <input name="production_date" 
                            type="date" 
                            class="form-control" 
                            value="<?= htmlspecialchars($_POST['production_date'] ?? $default_production_date) ?>"
                            <?= $production_date_readonly ?>>
                    <button type="submit" class="btn btn-primary btn-submit-box" title="حفظ البيانات">
                        <i class="bi bi-arrow-return-left"></i>
                    </button>
                </div>
            </div>
            
            <div class="col-md-6 mb-2">
                <label class="form-label">ملاحظات</label>
                <div class="input-group">
                    <textarea name="notes" class="form-control"><?= htmlspecialchars($_POST['notes'] ?? '') ?></textarea>
                    <button type="submit" class="btn btn-primary btn-submit-box" title="حفظ البيانات">
                        <i class="bi bi-arrow-return-left"></i>
                    </button>
                </div>
            </div>
        </div>
        
        <button type="submit" class="btn btn-primary mt-2">حفظ المنتج</button>
    </form>
    <?php else: ?>
        <div class="alert alert-info">صلاحيتك تسمح بالعرض فقط، لا يمكنك إضافة منتجات.</div>
    <?php endif; ?>
</div>
        
<?php 
// تعريف الألوان الفاتحة للأعمدة
$col_colors = [
    'id' => '#f8f9fa',
    'barcode' => '#e6f0ff',
    'name' => '#ffffff',
    'size' => '#e6fffd',
    'production_date' => '#fffbe6',
    'expiry_date' => '#ffebeb', 
    'notes' => '#f5f5f5',
    'created_by' => '#e6f7ff',
    'created_at' => '#e6e6ff',
    'delivery_note_ref' => '#d0e0fb',
    'delivery_date' => '#e6ffef',
];
?>

<div class="card">
    <h4 class="mb-3">قائمة المنتجات (<span class="text-primary"><?= $total_products ?></span>) - الصفحة الحالية: <?= $current_page ?></h4>
    <div class="table-responsive">
        <table class="table table-hover table-sm">
            
            <thead style="font-size: 0.9rem;">
                <tr style="border-bottom: 3px solid #ccc;">
                    <th scope="col" class="text-white text-center" style="width: 50px; background-color: #6c757d !important;">ID</th>
                    <th scope="col" class="text-white text-center" style="background-color: #0d6efd !important;">الباركود</th>
                    <th scope="col" class="text-white" style="background-color: #198754 !important;">الاسم</th>
                    <th scope="col" class="text-white text-center" style="width: 80px; background-color: #20c997 !important;">المقاس</th>
                    <th scope="col" class="text-center" style="background-color: #ffc107 !important; color: #333 !important;">تاريخ انتاج</th>
                    <th scope="col" class="text-white text-center" style="background-color: #007bff !important;">رقم السند</th>
                    <th scope="col" class="text-white text-center" style="background-color: #28a745 !important;">تاريخ التسليم</th>
                    <th scope="col" class="text-white" style="background-color: #6f42c1 !important;">الملاحظات</th>
                    <th scope="col" class="text-center" style="background-color: #0dcaf0 !important; color: #333 !important;">بواسطة</th>
                    <th scope="col" class="text-white text-center" style="background-color: #000080 !important;">تاريخ الإنشاء</th>
                    <th scope="col" class="text-white text-center" style="width: 120px; background-color: #343a40 !important;">إجراءات</th>
                </tr>
            </thead>
            
            <tbody>
                <?php 
                $user_list = $pdo->query("SELECT id, username FROM users")->fetchAll(PDO::FETCH_KEY_PAIR);
                
                foreach($rows as $r): 
                    $row_bg_color = ''; 
                ?>
                <tr style="<?= $row_bg_color ?>">
                    <td class="small text-center" style="background-color: <?= $col_colors['id'] ?> !important;"><?= $r['id'] ?></td>
                    <td class="small text-center text-break" style="background-color: <?= $col_colors['barcode'] ?> !important;"><strong><?= htmlspecialchars($r['barcode']) ?></strong></td>
                    <td class="small" style="background-color: <?= $col_colors['name'] ?> !important;"><?= htmlspecialchars($r['name']) ?></td>
                    <td class="small text-center" style="background-color: <?= $col_colors['size'] ?> !important;"><?= htmlspecialchars($r['size'] ?? '-') ?></td>
                    <td class="small text-center" style="background-color: <?= $col_colors['production_date'] ?> !important;"><?= htmlspecialchars($r['production_date'] ?? '-') ?></td>
                    
                    <td class="small text-center fw-bold" style="background-color: <?= $col_colors['delivery_note_ref'] ?> !important;">
                        <?= htmlspecialchars($r['delivery_note_ref'] ?? '-') ?>
                    </td>
                    <td class="small text-center" style="background-color: <?= $col_colors['delivery_date'] ?> !important;">
                        <?= htmlspecialchars($r['delivery_date'] ?? '-') ?>
                    </td>
                    
                    <td class="small" style="background-color: <?= $col_colors['notes'] ?> !important;">
                        <div class="text-truncate" style="max-width: 150px;" title="<?= htmlspecialchars($r['notes'] ?: 'لا توجد ملاحظات') ?>">
                            <?= htmlspecialchars($r['notes'] ?? '-') ?>
                        </div>
                    </td>
                    
                    <td class="small text-center text-muted" style="background-color: <?= $col_colors['created_by'] ?> !important;"><?= htmlspecialchars($user_list[$r['created_by']] ?? 'N/A') ?></td>
                    <td class="small text-center text-muted" style="background-color: <?= $col_colors['created_at'] ?> !important;"><?= date('Y-m-d', strtotime($r['created_at'])) ?></td>
                    
                    <td class="text-center" style="background-color: #f8f9fa !important;">
                        <?php if ($_SESSION['user']['role'] === 'super_admin'): ?>
                            <a href="edit_product.php?id=<?= $r['id'] ?>" class="btn btn-sm btn-info me-1 py-0">تعديل</a>
                            <button onclick="deleteProduct(<?= $r['id'] ?>)" class="btn btn-sm btn-danger py-0">حذف</button>
                        <?php else: ?>
                            <a href="edit_product.php?id=<?= $r['id'] ?>" class="btn btn-sm btn-info me-1 py-0">عرض</a>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach;?>
            </tbody>
        </table>
    </div>
    
    <?php if ($total_pages > 1): ?>
    <nav aria-label="Products Pagination">
        <ul class="pagination justify-content-center mt-4 flex-wrap">
            
            <li class="page-item <?= ($current_page <= 1) ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=1">الصفحة الأولى</a>
            </li>

            <li class="page-item <?= ($current_page <= 1) ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= max(1, $current_page - 1) ?>" aria-label="Previous">
                    <span aria-hidden="true">&laquo;</span>
                </a>
            </li>
            
            <?php
            // تحديد نطاق الأرقام المعروضة (مثلاً، 5 صفحات حول الصفحة الحالية)
            $range = 2; // عرض صفحتين قبل وبعد
            $start_loop = max(1, $current_page - $range);
            $end_loop = min($total_pages, $current_page + $range);
            
            if ($start_loop > 1) { echo '<li class="page-item disabled"><span class="page-link">...</span></li>'; }
            
            for ($i = $start_loop; $i <= $end_loop; $i++): 
            ?>
                <li class="page-item <?= ($i == $current_page) ? 'active' : '' ?>">
                    <a class="page-link" href="?page=<?= $i ?>"><?= $i ?></a>
                </li>
            <?php endfor;
            
            if ($end_loop < $total_pages) { echo '<li class="page-item disabled"><span class="page-link">...</span></li>'; }
            ?>
            
            <li class="page-item <?= ($current_page >= $total_pages) ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= min($total_pages, $current_page + 1) ?>" aria-label="Next">
                    <span aria-hidden="true">&raquo;</span>
                </a>
            </li>
            
            <li class="page-item <?= ($current_page >= $total_pages) ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= $total_pages ?>">الصفحة الأخيرة</a>
            </li>
        </ul>
    </nav>
    <?php endif; ?>

</div>
    
    <script src="https://unpkg.com/@zxing/library@0.18.6/umd/index.min.js"></script>
    <script>
        // 1. كود الحذف (deleteProduct)
        async function deleteProduct(id) {
            if (!confirm("هل أنت متأكد من حذف المنتج رقم " + id + "؟")) {
                return;
            }
            try {
                const res = await fetch('/barcode-system/api/products.php', {
                    method: 'DELETE',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'id=' + id
                });
                const json = await res.json();
                if (json.ok) {
                    alert("تم حذف المنتج بنجاح.");
                    window.location.reload();
                } else {
                    alert("خطأ في الحذف: " + (json.error || "خطأ غير معروف"));
                }
            } catch (e) {
                alert("حدث خطأ في الاتصال بالخادم.");
            }
        }
        
        // 2. كود المسح الضوئي (مع تحديث بسيط للأيقونات)
        const codeReader = new ZXing.BrowserBarcodeReader();
        const video = document.getElementById('video');
        const scanBtn = document.getElementById('scanBtn');
        const barcodeInput = document.getElementById('barcodeInput');
        const scanBtnIcon = scanBtn.querySelector('i'); 
        let active = false;

        scanBtn.addEventListener('click', () => {
            if (active) {
                // إيقاف الكاميرا
                codeReader.reset();
                video.style.display = 'none';
                scanBtnIcon.className = 'bi bi-camera-fill';
                active = false;
                return;
            }

            // بدء الكاميرا
            video.style.display = 'block';
            scanBtnIcon.className = 'bi bi-camera-video-fill';
            active = true;

            codeReader.decodeFromVideoDevice(undefined, video, (result, err) => {
                if (result) {
                    barcodeInput.value = result.text.trim();
                    
                    // إيقاف الكاميرا فوراً بعد القراءة
                    codeReader.reset();
                    video.style.display = 'none';
                    scanBtnIcon.className = 'bi bi-camera-fill';
                    active = false;
                    
                    // 🟢 تفعيل الحفظ بعد المسح
                    document.getElementById('addProductForm').submit();
                }

                if (err && !(err instanceof ZXing.NotFoundException)) {
                    console.error(err);
                }
            });
        });

    </script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>